#ifndef __CGraphics__
#define __CGraphics__

#include <MathTools/CPoint.hpp>
#include <MathTools/CRect.hpp>
#include <MathTools/CDimension.hpp>
#include <MathTools/CPolygon.hpp>
#include <MathTools/CMathTools.hpp>
#include <Testing/CTrace.hpp>
#include "../Windowing/CClip.hpp"
#include "../Fonts/CFont.hpp"
#include "IImage.hpp"
#include "CPen.hpp"
#include "CBrush.hpp"
#include "CNativeImage.hpp"

//	===========================================================================

using Exponent::MathTools::CPoint;
using Exponent::MathTools::CRect;
using Exponent::MathTools::CDimension;
using Exponent::MathTools::CPolygon;
using Exponent::MathTools::CMathTools;
using Exponent::Testing::CTrace;
using Exponent::GUI::Graphics::IImage;
using Exponent::GUI::Graphics::CPen;
using Exponent::GUI::Graphics::CBrush;
using Exponent::GUI::Graphics::CNativeImage;
using Exponent::GUI::Windowing::CClip;
using Exponent::GUI::Fonts::CFont;

//	===========================================================================

#ifdef WIN32
	#pragma comment (lib, "Msimg32.lib")
#endif

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Graphics
		{
			/**
			 * @class CGraphics CGraphics.hpp
			 * @brief Graphics context - All drawing is actually done in this class
			 *
			 * Graphics encapsulates the drawing capabilities of the platform. Whilst every effort has been made to make sure that functions work identically on both mac and windows\n
			 * there are some areas where they dont align exactly. Internally the graphics system maintains a seperate 'back buffer' to enable the double buffering drawing to take place.\n
			 * All graphics draws are performed in to this back buffer, and then blitted in one operation to the main graphic context that the user can view.\n
			 * Each graphics context holds pointers to both a pen and a brush that are used for the drawing operations. The pen is used to draw lines, the brush for drawing block areas of colour.\n
			 * The graphics object also maintains both a clip area and a current offset. Because controls draw relative to their top left we need to maintain internally an offset that id added to all\n
			 * drawing co-ordinates. This assumes that the top left of a window has coordinates 0,0 with the x and y increasing along their axis (so a window that is 800x600 would have a bottom right of...800,600)\n
			 * The clipping area can be set to any area inside the total area of the graphics (as it is initialised with). However, not that this may mean that your drawing calls wont show up!\n
			 * Finally it should be pointed out that generally you should be creating, destroying or altering a CGraphics object. This should be handled internally by any window that needs to do drawing.
			 *
			 * @see CBrush
			 * @see CPen
			 * @see CWindow
			 * @see CControl
			 * @see CControlRoot
			 * @see CClip
			 *
			 * @date 01/10/2004
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @version 1.0.1 Added drawFilledBorderedRectangle function
			 * @version 1.0.2 Moved windows format to GDI+
			 * @version 1.0.3 Made text colour alpha based
			 *
			 * @todo Update more of the graphics context to GDI+ on windows. 
			 * @todo Add extra functions and remove any deprecated function
			 *
			 * @note On windows this class requires the use of the GDI+ library (gdiplus.lib). 
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CGraphics.hpp,v 1.14 2007/02/08 21:07:54 paul Exp $
			 */
			class CGraphics
			{
			public:

//	===========================================================================

				/**
				 * @enum ETextAlignment
				 * @brief Text alignment
				 */
				enum ETextAlignment
				{
					e_leftTopAlign = 0,				/**< Align the text to the top left of the specified area */
					e_centerTopAlign,				/**< Align the text to the center of the area and the top of the area */
					e_rightTopAlign,				/**< Align the text to the top right of the specified area */
					e_leftCenterAlign,				/**< Align the text to the center of the area and the left hand side */
					e_centerCenterAlign,			/**< Align the text centrally within the given area */
					e_rightCenterAlign,				/**< Align the text to the center of the area and the right hand side */
					e_leftBottomAlign,				/**< Align the text to the bottom left of the specified area */
					e_centerBottomAlign,			/**< Align the text to the center of the area and the bottom of the area */
					e_rightBottomAlign,				/**< Align the text to the bottom right of the specified area */
					e_defaultLabelText,				/**< Synonomous with e_leftCenterAlign */
					e_defaultSingleLine,			/**< Synonomous with e_centerCenterAlign */
				};

				/**
				 * Construction
				 */
				CGraphics();

				/**
				 * Destruction
				 */
				virtual ~CGraphics();

//	===========================================================================

				#ifdef WIN32
					/**
					 * Initialise
					 * @param windowHandle The handle of the window
					 * @param drawContext The windows draw contexr
					 * @param area The area of the drawing area
					 */
					void initialise(SWindowHandle *windowHandle, HDC drawContext, const CDimension &area);
				#else
					/**
					 * Initialise
					 * @param windowHandle The handle of the window
					 * @param drawContext The mac draw contexr
					 * @param area The area of the drawing area
					 */
					void initialise(SWindowHandle *windowHandle, CGContextRef drawContext, const CDimension &area);
				#endif

				/**
				 * Uninitialise
				 */
				void uninitialise();

//	===========================================================================

				/**
				 * Begin alpha drawing
				 * @param alpha The alpha level you want to draw with, 0 = fully transparent 0.5 = 50/50 blend src and dest 1.0 = fully opaque
				 * @note These functions are for large scale alpha blending (for example if you have rendered a whole interface to the context\n
				 * and you now want to blend it back to the main display. Note that all 'Fill / Draw' functions will independantly alpha blend based on the colour\n
				 * That is supplied to it.\n
				 * How to use alpha blending in the exponent system\n
				 * The alpha blending sub system operates in similar ways externally, but internally is very different between the platforms\n
				 * windows doesnt understand the concept of having an alpha level applied to its graphics context. For once the mac is better! :)\n
				 * Example draw function\n
				 * @\n
				 * @code
				 * void drawControl(CGraphics &graphics)
				 * {
				 *     // Area of each square relative to our top left
				 *     const CRect area[] =
				 *     {
				 *          CRect(10, 20, 50, 50),
				 *          CRect(20, 30, 50, 50),
				 *          CRect(30, 40, 50, 50)
				 *     };
				 *
				 *     // We want to draw three squares, each alpha blended differently over the top of the others
				 *     for (long i = 0; i < 3; i++)
				 *     {
				 *         // First we want to know where to draw to within the main native image
				 *         const CRect drawToHereRelativeToMyTopLeft(0, 0, m_area.getWidth(), m_area.getHeight());
				 *
				 *         // Next we want to now if there is an offset within the image we are going to alphablend. Use this to move the top left of the alpha blended image
				 *         const CRect offset(0, 0, m_area.getWidth(), m_area.getHeight());
				 *
				 *         // Call this before each alpha blend operation you want to make
				 *         graphics.beginAlphaDrawing(0.8);
				 *
				 *		   // Set the pen colour or whatever else here
				 *
				 *         graphics.fillRectangle(area[i]);
				 *
				 *         // Make sure you call this after *each* drawing operation. Not at the end of your entire draw function, else you wont get alpha blending!\n
				 *         graphics.endAlphaDrawing(drawToHereRelativeToMyTopLeft, offset);
				 *     }
				 * }
				 * @endcode
				 * @see endAlphaDrawing
				 */
				void beginAlphaDrawing(const double alpha);

				/**
				 * End the alpha drawing
				 * @param drawTo The area To draw to. Normally within a control that draws within itself, you would make this CRect(0, 0, w, h)
				 * @param offset The offset drawing position within the native window. Normally within a control that draws within itself, you would make this CRect(0, 0, w, h)
				 * @see beginAlphaDrawing
				 */
				void endAlphaDrawing(const CRect &drawTo = CRect(0, 0, 0, 0), const CRect &offset = CRect(0, 0, 0, 0));

//	===========================================================================

				/**
				 * Set the background colour to draw with
				 * @param colour The draw area background colour
				 */
				void setBackgroundColour(const CColour &colour) { m_nativeImage->setBackgroundColour(colour); }

//	===========================================================================

				/**
				 * Draw a single pixel at the point specified
				 * @param point The position to draw the point
				 */
				void drawPixel(const CPoint &point);

				/**
				 * Draw an anti aliased pixel
				 * @param point The position to draw the point
				 * @param alpha The alpha value to use
				 */
				void drawAntiAliasedPixel(const CPoint &point, const long alpha);

				/**
				 * Move the pen to a location
				 * @param point The point to move to
				 */
				void moveTo(const CPoint &point);

				/**
				 * Draw a line
				 * @param start The start position
				 * @param end The end position
				 */
				void drawLine(const CPoint &start, const CPoint &end);

				/**
				 * Draw a line from the current position to this position
				 * @param toThisPoint Position to end drawing at
				 */
				void drawLine(const CPoint &toThisPoint);

				/**
				 * Draw an anti aliased line
				 * @param start The start position
				 * @param end The end position
				 */
				void drawAntiAliasedLine(const CPoint &start, const CPoint &end);

//	===========================================================================

				/**
				 * Draw a rectangle outline
				 * @param area The area to to outline
				 */
				void drawRectangle(const CRect &area);

				/**
				 * Draw a rounded rectangle
				 * @param rect The rect to fill
				 * @param roundingEffect The amount of rounding to apply
				 */
				void drawRoundedRectangle(const CRect &rect, const CDimension &roundingEffect);

				/**
				 * Draw a polygon
				 * @param polygon The polygon
				 */
				void drawPolygon(const CPolygon &polygon);

				/**
				 * Draw an anti aliased polygon
				 * @param polygon The polygon
				 */
				void drawAntiAliasedPolygon(const CPolygon &polygon);

				/**
				 * Draw an ellipse
				 * @param area The bounding rectangle
				 */
				void drawEllipse(const CRect &area);

				/**
				 * Draw an arc
				 * @param area The bounding rectangle
				 * @param startPos The starting position around the circle
				 * @param endPos The ending position around the circle
				 * @param clockwise Line draw clockwise if true, anticlockwise if false
				 */
				void drawArc(const CRect &area, const CPoint &startPos, const CPoint &endPos, const bool clockwise = true);

//	===========================================================================

				/**
				 * fill a rectange with the current brush
				 * @param rect The rect to fill
				 */
				void fillRectangle(const CRect &rect);

				/**
				 * Fill a rounded rectangle with the current brush
				 * @param rect The rect to fill
				 * @param roundingEffect The amount of rounding to apply
				 */
				void fillRoundedRectangle(const CRect &rect, const CDimension &roundingEffect);

				/**
				 * Fill an ellipse with the current brush
				 * @param rect The bounding rectangle
				 */
				void fillEllipse(const CRect &rect);

				/**
				 * Fill a polygon
				 * @param polygon The polygon
				 */
				void fillPolygon(const CPolygon &polygon);

//	===========================================================================

				/**
				 * Draw a filled rectangle with a border
				 * @param area The are to draw
				 */
				void drawFilledBorderedRectangle(const CRect &area);

				/**
				 * Draw a gradient filled rectangle. Fades from current pen colour to white in the middle
				 */
				void drawGradientFilledRectangle(const CRect &area);

//	===========================================================================

				/**
				 * Draw an image
				 * @param image The image to draw
				 * @param drawTo The area to draw to
				 * @param offset The offset within the original object
				 */
				void drawImage(IImage *image, const CRect &drawTo, const CRect &offset);

				/**
				 * Draw an image with alpha
				 * @param image The image to draw
				 * @param drawTo The area to draw to
				 * @param offset The offset within the original object
				 * @param alpha The alpha level to draw with
				 */
				void drawImageWithAlpha(IImage *image, const CRect &drawTo, const CRect &offset, const double alpha);

//	===========================================================================

				/**
				 * Draw some text
				 * @param text The text to draw
				 * @param area The area to draw in
				 * @param theFont The font to draw with
				 * @param drawOptions The drawing options to use
				 */
				void drawText(const CString &text, const CRect &area, const CFont *theFont, const ETextAlignment drawOptions = e_defaultLabelText);

				/**
				 * Set the text colour
				 * @param textColour The text colour to use
				 */
				void setTextColour(const CAlphaColour &textColour = CAlphaColour::CALPHACOLOUR_BLACK) { m_textColour = textColour; }

//	===========================================================================

				/**
				 * Get the pen
				 * @retval CPen* The pen
				 */
				CPen *getMutablePen() const;

				/**
				 * Get the pen
				 * @retval const CPen* The pen
				 */
				const CPen *getPen() const;

				/**
				 * Get the brush
				 * @retval CBrush* The brush
				 */
				CBrush *getMutableBrush() const;

				/**
				 * Get the brush
				 * @retval const CBrush* The brush
				 */
				const CBrush *getBrush() const;

				/**
				 * Get the native image This is the back buffer and you should not be messing with this function directly unless you absolutely know what you are doing
				 * @retval const CNativeImage* The native drawing image
				 */
				const CNativeImage *getNativeImage() const { return m_nativeImage; }

				/**
				 * Get the native image. This is the back buffer and you should not be messing with this function directly unless you absolutely know what you are doing
				 * @retval CNativeImage* The native drawing image
				 */
				CNativeImage *getMutableNativeImage() const { return m_nativeImage; }

//	===========================================================================

				/**
				 * Offset the areas for drawing
				 * @param point The point to offset to
				 */
				void setDrawingAreaOffset(const CPoint &point) { m_graphicsOffset = point; }

				/**
				 * Append to the offset for drawing
				 * @param point The point to offset by
				 */
				void appendToDrawingAreaOffset(const CPoint &point) { m_graphicsOffset.offset(point.getXPosition(), point.getYPosition()); }

				/**
				 * Netaget from the offset to drawing
				 * @param point The point to remove from drawing offset
				 */
				void negateFromDrawingArea(const CPoint &point)  { m_graphicsOffset.offset(-point.getXPosition(), -point.getYPosition()); }

				/**
				 * Reset the areas for drawing, all drawing begins at (0,0) again
				 */
				void resetDrawingArea() { m_graphicsOffset.setPoint(0, 0); }

				/**
				 * Get the drawing area offset
				 * @retval const CPoint& The amount the initial pen position (0,0) is nmoved by
				 */
				const CPoint &getDrawingAreaOffset() const { return m_graphicsOffset;}

//	===========================================================================

				/**
				 * Set the update area
				 * @param area The global update area in pixels
				 */
				void setUpdateArea(const CRect &area) { m_updateArea = area; }

				/**
				 * Get the update area
				 * @retval const CRect& The update area in global pixels
				 */
				const CRect &getUpdateArea() const { return m_updateArea; }

//	===========================================================================

				/**
				 * Get the clipping region
				 * @retval CClip& The clipping area
				 */
				CClip &getClippingRegion();

//	===========================================================================

			protected:

//	===========================================================================

				#ifdef WIN32
					/**
					 * Draw an AA line using wu algorithm
					 * @param start The start point
					 * @param end The end point
					 */
					void drawWUAALine(const CPoint &start, const CPoint &end);

//	===========================================================================

					/**
					 * Draw an antialiased pixel with constantly increasing x position
					 * @param x The x position
					 * @param y The y position
					 * @param alpha The alpha value
					 */
					FORCEINLINE void drawAAPixelIncreasingX(const long x, const long y, const unsigned long alpha)
					{
						CPoint myPoint(x, y);
						myPoint.offset(m_graphicsOffset);

						// Store positions
						const long xx = myPoint.getXPosition();
						const long yy = myPoint.getYPosition();

						// First we need to get the colour of the pixel to be overwritten
						static CColour backgroundColour[3];			// Make these true statics
						static CAlphaColour blend[3];
						static COLORREF colours[3] = { GetPixel(m_drawContext, xx, yy), GetPixel(m_drawContext, xx + 1, yy), GetPixel(m_drawContext, xx - 1, yy) };

						// Get all the alpha pixels
						backgroundColour[0].setFromColourRef(colours[0]);
						backgroundColour[1].setFromColourRef(colours[1]);
						backgroundColour[2].setFromColourRef(colours[2]);
						CAlphaColour current = m_pen->getColour();

						// Now we do the actual alpha blending
						CAlphaColour::alphaBlend(blend[0], backgroundColour[0], current, 255);
						CAlphaColour::alphaBlend(blend[1], backgroundColour[1], current, alpha);
						CAlphaColour::alphaBlend(blend[2], backgroundColour[2], current, 255 - alpha);

						// Draw the three pixels
						m_pen->setColour(blend[0]);
						SetPixelV(m_drawContext, xx, yy, blend[0].getAsColourRef());

						m_pen->setColour(blend[1]);
						SetPixelV(m_drawContext, xx + 1, yy, blend[1].getAsColourRef());

						m_pen->setColour(blend[2]);
						SetPixelV(m_drawContext, xx - 1, yy, blend[2].getAsColourRef());

						// Resotre the old colour
						m_pen->setColour(current);
					}

					/**
					 * Draw an antialiased pixel with constantly increasing y position
					 * @param x The x position
					 * @param y The y position
					 * @param alpha The alpha value
					 */
					FORCEINLINE void drawAAPixelIncreasingY(const long x, const long y, const unsigned long alpha)
					{
						CPoint myPoint(x, y);
						myPoint.offset(m_graphicsOffset);

						// Store positions
						const long xx = myPoint.getXPosition();
						const long yy = myPoint.getYPosition();

						// First we need to get the colour of the pixel to be overwritten
						static CColour backgroundColour[3];			// Make these true statics
						static CAlphaColour blend[3];
						static COLORREF colours[3] = { GetPixel(m_drawContext, xx, yy), GetPixel(m_drawContext, xx, yy + 1), GetPixel(m_drawContext, xx, yy - 1) };

						// Get all the alpha pixels
						backgroundColour[0].setFromColourRef(colours[0]);
						backgroundColour[1].setFromColourRef(colours[1]);
						backgroundColour[2].setFromColourRef(colours[2]);
						CAlphaColour current = m_pen->getColour();

						// Now we do the actual alpha blending
						CAlphaColour::alphaBlend(blend[0], backgroundColour[0], current, 255);
						CAlphaColour::alphaBlend(blend[1], backgroundColour[1], current, alpha);
						CAlphaColour::alphaBlend(blend[2], backgroundColour[2], current, 255 - alpha);

						// Draw the three pixels
						m_pen->setColour(blend[0]);
						SetPixelV(m_drawContext, xx, yy, blend[0].getAsColourRef());

						m_pen->setColour(blend[1]);
						SetPixelV(m_drawContext, xx, yy + 1, blend[1].getAsColourRef());

						m_pen->setColour(blend[2]);
						SetPixelV(m_drawContext, xx, yy - 1, blend[2].getAsColourRef());

						// Resotre the old colour
						m_pen->setColour(current);
					}

					/**
					 * @struct SAlphaGraphics CGraphics.hpp
					 * @brief Small struct to store all of the alpha values required
					 */
					struct SAlphaGraphics
					{
						CPen *m_oldPen;					/**< The pen we replaced with alpha one */
						CBrush *m_oldBrush;				/**< The brush we replaced with alpha one */
						double m_alphaValue;			/**< The final alpha value */
						CNativeImage *m_alphaImage;		/**< The alpha blend image */
						CClip m_alphaClip;				/**< The alpha clip rect */
					};
				#endif

//	===========================================================================

				CPen *m_pen;										/**< The pen for drawing with.. */
				CBrush *m_brush;									/**< The brush */
				CNativeImage *m_nativeImage;						/**< The double buffer */
				CClip m_clip;										/**< The clipping region */

				CPoint m_graphicsOffset;							/**< Amount of offset to apply */
				CRect m_updateArea;									/**< The update area */
				CDimension m_rootArea;								/**< The total area (the size of the window this graphics is drawing to */

				CAlphaColour m_textColour;							/**< The colour of text drawn onscreen */

				bool m_isAlphaBlending;								/**< Are we currently alpha blending? */

				#ifdef WIN32
					HDC m_drawContext;								/**< The windows draw context */
					SAlphaGraphics m_alphaGraphics;					/**< The alpha graphics components */
					bool m_hasGdiPlus;								/**< Does the system support GDI+ */
					ULONG_PTR m_gdiPlusToken;						/**< Token for GDI+ */
				#else
					CGContextRef m_theContext;						/**< The context for drawing in to */
				#endif

//	===========================================================================

			};
		}
	}
}
#endif	// End of CGraphics.hpp